<?php

/**
 * Registration Entries List
 *
 * @package    Wp_Events/admin
 * @subpackage Wp_Events/admin/includes
 * @since      1.0.500
 */

// If this file is called directly, abort.
if ( !defined('WPINC') ) {
	die;
}

/**
 * Includes WP_List_Table class from wordpress core.
 */
if ( !class_exists('WP_List_Table') ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

/**
 * New class that extends WP_List_Table, displays form entries
 * and handles user actions.
 *
 * @since 1.0.449
 * 
 * @see WP_List_Table
 */
class Wp_Events_Registrations_list extends WP_List_Table {

	/**
	 * Name of the table in database.
	 * 
	 * @since 1.0.449
	 * @var string $table_name
	 */
	protected $table_name = 'events_registration';

	/**
	 * Class Constructor.
	 * 
	 * Calls constructor from the parent class.
	 *
	 * @since 1.0.449
	 */
	public function __construct() {
		parent::__construct([
			'singular' => __( 'Registrations', 'simple-wp-events' ),
			//singular name of the listed records
			'plural'   => __( 'Registrations', 'simple-wp-events' ),
			//plural name of the listed records
			'ajax'     => FALSE
			//should this table support ajax?
		]);
	}

	/**
	 * Message to display when no items are available.
	 * 
	 * @since 1.0.449
	 */
	public function no_items() {
		esc_html_e( 'No Registrations available.', 'simple-wp-events' );
	}

	/**
	 * Define what data to show on each column of the table.
	 *
	 * @param  Array   $item         Data.
	 * @param  String  $column_name  - Current column name.
	 *
	 * @since 1.0.449
	 * @return Mixed
	 */
	public function column_default( $item, $column_name ) {

		switch ( $column_name ) {
			case 'first_name':
			case 'last_name':
			case 'email':
				return $item->$column_name;
			case 'event':
				return '<a href="' . get_the_permalink( $item->post_id ) . '">' . get_the_title( $item->post_id ) . '</a>';
			case 'event_type':
				return get_post_meta( esc_attr( $item->post_id ), 'wpevent-type', TRUE );
			case 'time':
				return $item->time_generated;
			case 'wpe_seat':
				return $item->wpe_seats;
			case 'texting_permission': 
				return $item->$column_name;
			case 'ID':
				return $item->ID;
			default:
				// Safe fallback for unknown columns
				if ( property_exists( $item, $column_name ) ) {
					return esc_html( $item->$column_name );
				}
				return '—'; // Return dash for unknown columns
		}
	}

	/**
	 * Render the bulk edit checkbox.
	 *
	 * @param  array  $item
	 *
	 * @since 1.0.449
	 * @return string
	 */
	function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="bulk-delete[]" value="%s" />',
			$item->ID
		);
	}

	/**
	 * Returns an associative array containing the bulk action.
	 *
	 * @return array
	 * @since 1.1.0
	 */
	public function get_bulk_actions() {


		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		$bulk_actions['bulk-delete'] = __( 'Move to Trash', 'simple-wp-events' );
		$display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all'; 
	
		switch ( $display ) {
			case 'pending':
				$bulk_actions['approve-entry'] = __( 'Approve', 'simple-wp-events' );
				$bulk_actions['cancel-entry']  = __( 'Cancel', 'simple-wp-events' );
				break;
			case 'trash':
				unset( $bulk_actions['bulk-delete'] );
				$bulk_actions['permanent-delete'] = __( 'Delete Permanently', 'simple-wp-events' );
				$bulk_actions['restore']          = __( 'Restore', 'simple-wp-events' );
				break;
		}
	
		return $bulk_actions;
	}
	

	/**
	 * Outputs HTML to display bulk actions and filters.
	 * 
	 * @since 1.0.449
	 * @param $which
	 */
	public function display_tablenav( $which ) {

		?>
		<input type='hidden' name='_wpnonce' value='<?php echo esc_attr(wp_create_nonce('wp_events_entries')); ?>' />
		<div class="tablenav 
		<?php echo esc_attr( $which ); ?>">
		<?php if ( $this->has_items() ) : ?>
		<div class="alignleft actions bulkactions">
		<?php $this->bulk_actions( $which ); ?>
		</div>
		<?php
			endif;
			$this->extra_tablenav( $which );
			$this->pagination( $which );
		?>
		<br class="clear" />
		</div>
		<?php
	}

	/**
	 * Handles single row actions of deletion and restore.
	 *
	 * @param $item
	 *
	 * @return string
	 * @since 1.1.0
	 */
	function column_ID( $item ) {

		$display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$page    = isset( $_REQUEST['page'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$event_options = get_option( 'wpe_events_settings' );
		$actions = [];
	
		if ( isset( $event_options['approve_registrations'] ) ) { // Only for pending approval entries
			if ( $display === 'pending' || $item->wpe_status == WPE_PENDING || $display === 'cancelled' || $item->wpe_status == WPE_CANCELLED ) {
				$actions['approve_entry'] = sprintf(
					'<a href="edit.php?post_type=wp_events&page=%s&display=%s&action=%s&bulk-delete[0]=%s&action2=approve-entry&_wpnonce=%s">%s</a>',
					esc_attr( $page ),
					esc_attr( $display ),
					'approve-entry',
					esc_attr( $item->ID ),
					wp_create_nonce( 'wp_events_entries' ),
					__( 'Approve', 'simple-wp-events' )
				);
			}
			if ( $display === 'pending' || $item->wpe_status == WPE_PENDING || $display === 'approved' || $item->wpe_status == WPE_APPROVED ) {
				$actions['cancel_entry'] = sprintf(
					'<a href="edit.php?post_type=wp_events&page=%s&display=%s&action=%s&bulk-delete[0]=%s&action2=cancel-entry&_wpnonce=%s">%s</a>',
					esc_attr( $page ),
					esc_attr( $display ),
					'cancel-entry',
					esc_attr( $item->ID ),
					wp_create_nonce( 'wp_events_entries' ),
					__( 'Cancel', 'simple-wp-events' )
				);
			}
		}
	
		$eventID = '';
	
		// If there is more than one event with same name
		if ( isset( $_GET['wpe_titles'] ) && $_GET['wpe_titles'] !== '' ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$event_ids = explode( ',', sanitize_text_field( wp_unslash( $_GET['wpe_titles'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			foreach ( $event_ids as $event_id ) {
				if ( $event_id == $item->post_id ) {
					$eventID = '&event=' . esc_attr( $event_id );
					break;
				}
			}
		}
	
		$actions['view_entry'] = sprintf(
			'<a href="edit.php?post_type=wp_events&page=wpe_view_entry%s&entry=%s&tab=registrations&display=%s&_wpnonce=%s">%s</a>',
			$eventID,
			esc_attr( $item->ID ),
			esc_attr( $display ),
			wp_create_nonce( 'wp_events_entries' ),
			__( 'View', 'simple-wp-events' )
		);
	
		if ( $display === 'trash' ) { // Only for trash entries
			$actions['delete_permanent'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&display=trash&action=%s&bulk-delete[0]=%s&action2=permanent-delete&_wpnonce=%s" onclick="return confirm(\'Are you sure you want to delete item(s)?\');">%s</a>',
				esc_attr( $page ),
				'permanent-delete',
				esc_attr( $item->ID ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Delete Permanently', 'simple-wp-events' )
			);
	
			$actions['restore'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&display=trash&action=%s&bulk-delete[0]=%s&action2=restore&_wpnonce=%s">%s</a>',
				esc_attr( $page ),
				'restore',
				esc_attr( $item->ID ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Restore', 'simple-wp-events' )
			);
		} else {
			$actions['delete'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&display=%s&action=%s&bulk-delete[0]=%s&action2=bulk-delete&_wpnonce=%s">%s</a>',
				esc_attr( $page ),
				esc_attr( $display ),
				'bulk-delete',
				esc_attr( $item->ID ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Move To Trash', 'simple-wp-events' )
			);
		}
	
		return sprintf(
			'%1$s %2$s',
			esc_html( $item->ID ),
			$this->row_actions( $actions )
		);
	}
	

	/**
	 * Handles data query and filter, sorting, and pagination.
	 * 
	 * @since 1.0.449
	 */
	public function prepare_items() {

		if ( ! Wp_Events_Db_Actions::wpe_table_exists( $this->table_name ) ) {
			Wp_Events_Db_Actions::add_registration_table();
		}
	
		// Sanitize search key.
		$search_key = '';
		if ( isset( $_REQUEST['s'] ) ) {
			$search_key = sanitize_text_field( trim( wp_unslash( $_REQUEST['s'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		}
	
		$this->_column_headers = [
			$this->get_columns(),
			[], // hidden columns
			$this->get_sortable_columns(),
			$this->get_primary_column_name(),
		];
	
		// Delete all trash entries when empty trash button is clicked.
		if ( isset( $_GET['emptytrash'] ) && isset( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] )), 'wp_events_entries' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->wpe_empty_trash();
		}
	
		/** Process bulk action */
		$this->process_bulk_action();
	
		$per_page     = $this->get_per_page();
		$current_page = $this->get_pagenum();
	
		// Create SQL string for filter dropdowns
		$filter_string       = '';
		$filter_posts_array  = $this->get_filter_string();
	
		if ( isset( $filter_posts_array ) && ! empty( $filter_posts_array ) ) {
			$postID_string  = implode( ', ', array_map( 'intval', $filter_posts_array ) );
			$filter_string  = ' AND post_id in (' . $postID_string . ')';
		} elseif ( $filter_posts_array === [] ) {
			$filter_string = ' AND post_id in (-1)';
		}
	
		$total_items = $this->record_count( $filter_string );
	
		if ( $search_key ) {
			$table_data  = $this->get_event_registrations( -1, $current_page, $filter_string );
			$table_data  = $this->filter_table_data( $table_data, $search_key );
			$total_items = count( $table_data );
			$per_page    = $total_items;
		} else {
			$table_data = $this->get_event_registrations( $per_page, $current_page, $filter_string );
		}
	
		$this->set_pagination_args([
			'total_items' => $total_items,
			'per_page'    => $per_page,
		]);
	
		$this->items = $table_data;
	}
	
	

	/**
	 * Gets a list of columns.
	 *
	 * The format is:
	 * - `'internal-name' => 'Title'`
	 *
	 * @since 1.0.449
	 *
	 * @return array
	 */
	public function get_columns() {
		return [
			'cb'         			=> '<input type="checkbox">',
			'ID'         			=> __('Id', 'simple-wp-events'),
			'first_name' 			=> __('First Name', 'simple-wp-events'),
			'last_name'  			=> __('Last Name', 'simple-wp-events'),
			'email'      			=> __('Email', 'simple-wp-events'),
			'event'      			=> __('Event', 'simple-wp-events'),
			'event_type' 			=> __('Type', 'simple-wp-events'),
			'wpe_seat'   			=> __('Seats', 'simple-wp-events'),
			'texting_permission'	=> __('Texting Permission', 'simple-wp-events'),
			'time'       			=> __('Time', 'simple-wp-events'),
		];
	}

	/**
	 * Columns to make sortable.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {
		return [
			'ID'       => ['id', TRUE],
			'time'     => ['time_generated', TRUE],
			'wpe_seat' => ['wpe_seats', TRUE],
		];
	}

	/**
	 * Process bulk actions of deletion, restore, approve, cancel entries.
	 * 
	 * @since 1.2.0
	 */
	public function process_bulk_action() {

		$bulkaction	   = $this->current_action();
		$event_options = get_option('wpe_events_settings');

		switch ( $bulkaction ) {
			case 'bulk-delete':
				$this->wpe_process_bulk_action( WPE_TRASHED, __( ' item(s) moved to the Trash.', 'simple-wp-events' ) );
				break;
			case 'permanent-delete':
				$this->wpe_process_bulk_action( WPE_DELETED, __( ' item(s) permanently deleted.', 'simple-wp-events' ) );
				break;
			case 'cancel-entry':
				$this->wpe_process_bulk_action( WPE_CANCELLED,  __( ' item(s) Cancelled.', 'simple-wp-events' ) );
				$this->wpe_approve_cancel_notification( WPE_CANCELLED );
				break;
			case 'approve-entry':
				$this->wpe_process_bulk_action( WPE_APPROVED, __( ' item(s) Approved.', 'simple-wp-events' ) );
				$this->wpe_approve_cancel_notification( WPE_APPROVED );
				break;
			case 'restore':
				if ( isset( $event_options['approve_registrations'] ) ) {
					$this->wpe_process_bulk_action( WPE_PENDING, __( ' item(s) restored from the Trash.', 'simple-wp-events' ) );
				} else {
					$this->wpe_process_bulk_action( WPE_ACTIVE, __( ' item(s) restored from the Trash.', 'simple-wp-events' ) );
				}
		}
	}

	/**
	 * Trash/Delete, Approve/Cancel a customer record.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 *
	 * @param  int  $id  customer ID.
	 * @param int $val status of registration.
	 * 
	 * @since 1.2.0
	 *
	 * @return bool|int
	 */
	public function update_registration_status( $id, $val ) {
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->update(
			"{$wpdb->prefix}$this->table_name",
			['wpe_status' => $val],
			['id' => $id],
			'%d',
			'%d'
		);
	}

	/**
	 * Returns per page items from screen options.
	 * 
	 * @since 1.0.449
	 * 
	 * @return int 
	 */
	public function get_per_page() {
		// get the current user ID
		$user = get_current_user_id();
		// get the current admin screen
		$screen = get_current_screen();
		// retrieve the "per_page" option
		$screen_option = $screen->get_option( 'per_page', 'option' );
		// retrieve the value of the option stored for the current user
		$per_page = get_user_meta( $user, $screen_option, TRUE );
		if ( empty( $per_page ) || $per_page < 1 ) {
			// get the default value if none is set
			$per_page = $screen->get_option( 'per_page', 'default' );
		}

		return $per_page;
	}

	/**
	 * Returns the count of records (to display on screen) in the database.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 *
	 * @return null|string
	 */
	public function record_count( $filter_string ) {
		global $wpdb;
		
		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		// Securely fetch and sanitize $_GET['display']
		$display_tab = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all';
		$table = $wpdb->prefix . $this->table_name;
	
		$where_clause = '';
		$params       = [];
	
		switch ( $display_tab ) {
			case 'trash':
				$where_clause = 'wpe_status = %d';
				$params[]     = WPE_TRASHED;
				break;
			case 'approved':
				$where_clause = 'wpe_status = %d';
				$params[]     = WPE_APPROVED;
				break;
			case 'cancelled':
				$where_clause = 'wpe_status = %d';
				$params[]     = WPE_CANCELLED;
				break;
			case 'pending':
				$where_clause = 'wpe_status = %d';
				$params[]     = WPE_PENDING;
				break;
			default:
				$where_clause = 'wpe_status IN (%d, %d, %d, %d)';
				$params       = [ WPE_ACTIVE, WPE_PENDING, WPE_APPROVED, WPE_CANCELLED ];
				break;
		}
	
		// Prepare the full SQL query with optional additional filters
		$sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_clause} {$filter_string}";
	
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		return $wpdb->get_var( $wpdb->prepare( $sql, ...$params ) );
	}
	
	

	/**
	 * Retrieve customer’s data from the database.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 *
	 * @param  int  $per_page from screen options.
	 * @param  int  $page_number
	 * @param string $filter_string from title and category filters.
	 *
	 * @return mixed
     *
     * @since 1.1.0
	 */
	public function get_event_registrations( $per_page = 5, $page_number = 1, $filter_string = '' ) {
		global $wpdb;
	
		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		$table = $wpdb->prefix . $this->table_name;
		$display_tab = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all';
	
		$where_clause = '';
		$params = [];
	
		switch ( $display_tab ) {
			case 'trash':
				$where_clause = 'wpe_status = %d';
				$params[] = WPE_TRASHED;
				break;
			case 'approved':
				$where_clause = 'wpe_status = %d';
				$params[] = WPE_APPROVED;
				break;
			case 'cancelled':
				$where_clause = 'wpe_status = %d';
				$params[] = WPE_CANCELLED;
				break;
			case 'pending':
				$where_clause = 'wpe_status = %d';
				$params[] = WPE_PENDING;
				break;
			default:
				$where_clause = 'wpe_status IN (%d, %d, %d, %d)';
				$params = [ WPE_ACTIVE, WPE_PENDING, WPE_APPROVED, WPE_CANCELLED ];
				break;
		}
	
		// Add filter string if present
		$filter_sql = '';
		if ( ! empty( $filter_string ) ) {
			$filter_sql = " {$filter_string}";
		}
	
		$sql = "SELECT * FROM {$table} WHERE {$where_clause}{$filter_sql}";
	
		// Handle ordering
		if ( ! empty( $_REQUEST['orderby'] ) ) {
			$order_by = sanitize_sql_orderby( wp_unslash($_REQUEST['orderby']) );
			$order = ! empty( $_REQUEST['order'] ) ? sanitize_sql_orderby( wp_unslash($_REQUEST['order']) ) : 'DESC';
			$sql .= " ORDER BY {$order_by} {$order}";
		} else {
			$sql .= " ORDER BY time_generated DESC";
		}
	
		// Handle pagination
		if ( $per_page !== -1 ) {
			$offset = ( $page_number - 1 ) * $per_page;
			$sql .= " LIMIT %d OFFSET %d";
			$params[] = $per_page;
			$params[] = $offset;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		return $wpdb->get_results( $wpdb->prepare( $sql, ...$params ), OBJECT );
	}
	

	/**
	 * Filters the table data based on search key.
	 *
	 * @param array $table_data
	 * @param string $search_key
	 * 
	 * @since 1.0.449
	 * 
	 * @return bool
	 */
	public function filter_table_data( $table_data, $search_key ) {
		return array_values( array_filter( $table_data, function ( $row ) use ( $search_key ) {
			foreach ( $row as $row_val ) {
				if ( stripos( $row_val, $search_key ) !== FALSE ) {
					return TRUE;
				}
			}
		}));
	}

	/**
	 * Displays the list of views (all, trash) available on this table.
	 *
	 * @since 1.0.449
	 */
	public function views() {

		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		$wpe_current_display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all';

		$nonce = wp_create_nonce( 'wp_events_entries' );
		$views = [
			'all' => sprintf(
				'<a %shref="edit.php?post_type=wp_events&page=wp_forms_entries&display=all&_wpnonce=%s">%s</a>',
				wpe_is_current( $wpe_current_display, 'all' ),
				esc_attr( $nonce ),
				__( 'All', 'simple-wp-events' )
			),
		];
		
		$event_options = get_option('wpe_events_settings');
		$event 		   = '';
		if ( isset( $_GET['wpe_titles'] ) ) {
			$number = isset( $_GET['posts_page'] ) ? sanitize_text_field( wp_unslash( $_GET['posts_page'] ) ) : '1';
			$event  = '&posts_page=' . $number . '&wpe_titles=' . sanitize_text_field( wp_unslash( $_GET['wpe_titles'] ) );
		}

		if ( isset( $event_options['approve_registrations'] ) ) {
			$views['pending'] = sprintf(
				'<a %shref="edit.php?post_type=wp_events&page=wp_forms_entries&display=pending%s&_wpnonce=%s">%s</a>',
				wpe_is_current( $wpe_current_display, 'pending' ),
				esc_attr( $event ),
				esc_attr( $nonce ),
				__( 'Pending Approval', 'simple-wp-events' )
			);
			
			$views['approved'] = sprintf(
				'<a %shref="edit.php?post_type=wp_events&page=wp_forms_entries&display=approved%s&_wpnonce=%s">%s</a>',
				wpe_is_current( $wpe_current_display, 'approved' ),
				esc_attr( $event ),
				esc_attr( $nonce ),
				__( 'Approved', 'simple-wp-events' )
			);
			
			$views['cancelled'] = sprintf(
				'<a %shref="edit.php?post_type=wp_events&page=wp_forms_entries&display=cancelled%s&_wpnonce=%s">%s</a>',
				wpe_is_current( $wpe_current_display, 'cancelled' ),
				esc_attr( $event ),
				esc_attr( $nonce ),
				__( 'Cancelled', 'simple-wp-events' )
			);
		}

		$views['trash'] = sprintf(
			'<a %shref="edit.php?post_type=wp_events&page=wp_forms_entries&display=trash%s&_wpnonce=%s">%s</a>',
			wpe_is_current( $wpe_current_display, 'trash' ),
			esc_attr( $event ),
			esc_attr( $nonce ),
			__( 'Trash', 'simple-wp-events' )
		);

		/**
		 * Filters the list of available list table views.
		 *
		 * The dynamic portion of the hook name, `$this->screen->id`, refers
		 * to the ID of the current screen.
		 *
		 * @since 1.0.449
		 *
		 * @param string[] $views An array of available list table views.
		 */
		$views = apply_filters( "views_{$this->screen->id}", $views );

		if ( empty( $views ) ) {
			return;
		}

		echo "<ul class='subsubsub'>\n";
		$views_count = count( $views );
		$i = 0;
		foreach ( $views as $class => $view ) {
			$i++;
			echo "\t<li class='" . esc_attr( $class ) . "'>" . wp_kses_post( $view );
			if ( $i < $views_count ) {
				echo " |</li>\n";
			} else {
				echo "</li>\n";
			}
		}
		echo '</ul>';
		

	}

	/**
	 * Extra controls to be displayed between bulk actions and pagination.
	 * 
	 * Displays filters for event titles and categories.
	 *
	 * @since 1.0.449
	 *
	 * @param string $which
	 */
	protected function extra_tablenav( $which ) {
		if ( $which == "top" ) {
			$post_title_list = $this->output_titles_list();

			$this->empty_trash_button();

			if ( $post_title_list ) {
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Safe HTML output
				echo '<div class="alignleft actions bulkactions">' . $post_title_list . $this->output_categories_list() . $this->output_dates_filter(); ?>
					<input type="submit" id="filter-entries" class="button action" value="Filter">
					</div>
				<?php 
			}	
		}
	}

	/**
	 * Outputs HTML for event title dropdown filter.
	 * 
	 * @since 1.1.0
	 * @return string
	 */
	public function output_titles_list() {
		$post_title = null;
		$post_id	= null;
		$results 	= null;
		$args = array(
			'post_type' => 'wp_events'
		);

		$query = new WP_Query( $args );
		if ( $query->have_posts() ) :
			while ( $query->have_posts() ) : $query->the_post();
				$post_id[]	  = get_the_ID();
				$post_title[] = get_the_title();
			endwhile;
		endif;
		wp_reset_postdata();
		if (isset( $post_title ) && isset( $post_id ) ) {
			$results = wpe_array_combine( $post_title, $post_id );
		}

			// Return null if we found no results.
		if ( ! $results )
		return false;

		// HTML for our select printing post titles as loop.
		$output = '<select name="wpe_titles" id="wpe_titles" class="wpe-add-select2" searchable="Search here..">';

		$output .= '<option value="-1" selected>' . __( 'All Events', 'simple-wp-events' ) . '</option>';

		foreach ( $results as $title => $ids ) {
			if ( is_array( $ids ) ) {
				$ids = implode(',', $ids);
			} else {
				$ids = (string) $ids;
			}

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$selected = ( isset( $_GET['wpe_titles'] ) && ( (string) strpos( $ids, sanitize_text_field( wp_unslash( $_GET['wpe_titles'] ) ) )) !== '' ) ? "selected" : "";
			$output .= '<option value="' . $ids . '" ' . $selected . '>' . $title . '</option>';
		}

		$output .= '</select>'; // end of select element.

		// get the html.
		return $output;
	}

	/**
	 * Outputs HTML for categories dropdown filter.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 * 
	 * @since 1.0.449
	 * @return string
	 */
	public function output_categories_list() {
		$args = array(
			'taxonomy' => 'wpevents-category',
			'orderby' => 'name',
			'order'   => 'ASC'
		);

		$cats = get_categories( $args );

		if ( !$cats ) {
			return false;
		}

		// HTML for our select printing post categories as loop.
		$output = '<select name="wpe_categories" id="wpe_categories" class="mdb-select md-form" searchable="Search here..">';

		$output .= '<option value="-1">All Categories</option>';


		foreach ( $cats as $cat ) {
			$cat = (array) $cat;
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$selected = ( isset( $_GET['wpe_categories'] ) && $_GET['wpe_categories'] == $cat['slug']) ? "selected" : "";
			$output .= '<option value="' . $cat['slug'] . '" ' . $selected . '>' . $cat['name'] . '</option>';
		}

		$output .= '</select>'; // end of select element.

		// get the html.
		return $output;
	}

	/**
	 * Returns array of post IDs matching category filter.
	 *
	 * @since 1.0.449
	 * @return array
	 */
	public function get_categories_posts() {
		global $wpdb;
	
		$postID = null;
		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		if ( isset( $_GET['wpe_categories'] ) && $_GET['wpe_categories'] != -1 ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$category_slug = wpe_sanitize( sanitize_text_field( wp_unslash( $_GET['wpe_categories'] ) ) );
	
			// Get term ID from slug
			$term = get_term_by( 'slug', $category_slug, 'wpevents-category' );
			if ( $term && ! is_wp_error( $term ) ) {
				$term_id = intval( $term->term_id );
	
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$postIDs = $wpdb->get_col( $wpdb->prepare(
					"
					SELECT tr.object_id
					FROM {$wpdb->term_relationships} tr
					INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
					WHERE tt.term_id = %d
					AND tt.taxonomy = 'wpevents-category'
					",
					$term_id
				) );
	
				if ( $postIDs ) {
					$postID = array_map( 'strval', $postIDs );
				}
			}
		}
	
		return $postID;
	}
	

	/**
	 * Outputs HTML for start date and end date filter.
	 * 
	 * @since 1.1.0
	 * @return string
	 */
	public function output_dates_filter() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification.
		$start_date = isset( $_GET['wpe-filter-start-date'] ) ? sanitize_text_field( wp_unslash( $_GET['wpe-filter-start-date']  ) ) : '';
	
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Same as above.
		$end_date   = isset( $_GET['wpe-filter-end-date'] ) ? sanitize_text_field( wp_unslash( $_GET['wpe-filter-end-date'] )  ) : '';
	
		?>
			<input id="wpe-filter-start-date" autocomplete="off" class="wp-event-datepicker" type="text" name="wpe-filter-start-date" placeholder="Filter by start date" value="<?php echo esc_attr($start_date); ?>"/>
			<input id="wpe-filter-end-date" autocomplete="off" class="wp-event-datepicker" type="text" name="wpe-filter-end-date" placeholder="Filter by end date" value="<?php echo esc_attr($end_date); ?>"/>
		<?php
	}
	

	/**
	 * Returns array of post IDs matching dates filters.
	 *
	 * @since 1.1.12
	 * @return array
	 */
	public function get_date_filter_posts() {
		$post_ID = array();
		$args    = array(
			'post_type'      => 'wp_events',
			'posts_per_page' => '-1',
			'post_status'    => array( 'publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash' ),
		);
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification
		if ( isset( $_GET['wpe-filter-start-date'] ) && $_GET['wpe-filter-start-date'] !== '' && isset( $_GET['wpe-filter-end-date'] ) && $_GET['wpe-filter-end-date'] !== '' ) {
			$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
				'relation' => 'AND',
				[
					'key'     => 'wpevent-end-date-time',
					'compare' => '<=',
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification.
					'value'   => strtotime( sanitize_text_field( wp_unslash( $_GET['wpe-filter-end-date'] ) ) . '23:59:59' ),
					'type'    => 'numeric',
				],
				[
					'key'     => 'wpevent-start-date-time',
					'compare' => '>=',
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification.
					'value'   => strtotime( sanitize_text_field( wp_unslash( $_GET['wpe-filter-end-date'] ) ) ),
					'type'    => 'numeric',
				],
			];
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		} else if ( isset( $_GET['wpe-filter-start-date'] ) && $_GET['wpe-filter-start-date']  !== '' ) {
			$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
				[
					'key'     => 'wpevent-start-date-time',
					'compare' => '>=',
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification.
					'value'   => strtotime( sanitize_text_field( wp_unslash( $_GET['wpe-filter-start-date'] ) ) ),
					'type'    => 'numeric',
				],
			];
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		} else if ( isset( $_GET['wpe-filter-end-date'] ) && $_GET['wpe-filter-end-date'] !== '' ) {
			$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
				[
					'key'     => 'wpevent-end-date-time',
					'compare' => '<=',
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- These values are used for filtering only, not for data modification.
					'value'   => strtotime( sanitize_text_field(  wp_unslash( $_GET['wpe-filter-end-date'] ) ) . '23:59:59' ),
					'type'    => 'numeric',
				],
			];
		}
		$query = new WP_Query( $args );
			if ( $query->have_posts() ) :
				while ( $query->have_posts() ) : $query->the_post();
					$post_ID[] = (string) get_the_ID();
				endwhile;
			endif;
			wp_reset_postdata();
			return $post_ID;
	}

	/**
	 * Returns array of post IDs from all filters combined.
	 *
	 * @since 1.1.0
	 * @return array
	 */
	public function get_filter_string() {

		$category_filter = $this->get_categories_posts();
		$dates_filter 	 = $this->get_date_filter_posts();
		$filter			 = array();

		if ( isset( $category_filter ) && isset( $dates_filter ) ) {
			$filter = array_intersect( $category_filter, $dates_filter );
		} elseif ( isset( $category_filter ) ) {
			$filter = $category_filter;
		} elseif ( isset( $dates_filter ) ) {
			$filter = $dates_filter;
		}
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['wpe_titles'] ) && $_GET['wpe_titles'] != -1 ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$title_filter 	  = sanitize_text_field( wp_unslash( $_GET['wpe_titles']) );
			$title_filter_arr = explode( ",", $title_filter );
			if ( isset( $filter ) ) {
				$filter = array_intersect( $title_filter_arr, $filter );
			} else {
				$filter = $title_filter_arr;
			}
		}
		
		return $filter;
	}

	/**
	 * Generates custom admin notices on deletion/restore actions
	 * 
	 * @global string $pagenow 
	 *
	 * @param $message
	 * @since 1.1.0
	 */
	public function wpe_admin_notice( $message ) {
		global $pagenow;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( $pagenow == 'edit.php' && isset( $_REQUEST['page'] ) && 'wp_forms_entries' === $_REQUEST['page'] ) {
			 echo '<div class="notice notice-success is-dismissible">
				 <p>'. esc_html( $message ) .'</p>
			 </div>';
		}
	}

	/**
	 * Displays empty trash button if user is on trash tab and there are
	 * entries in the trash.
	 * 
	 * @since 1.1.0
	 */
	public function empty_trash_button() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['display'] ) && $_GET['display'] == 'trash' ) {
			echo '<input type="submit" name="emptytrash" class="button action" value="Empty Trash" 
				  onclick="return confirm(\'Are you sure you want to empty the Trash?\');" />';
		}
	}

	/**
	 * Permanently delete all entries from trash tab.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 * 
	 * @since 1.1.0
	 *
	 * @return bool|int
	 */
	public function wpe_empty_trash() {
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->update(
			"{$wpdb->prefix}$this->table_name",
			[ 'wpe_status' => WPE_DELETED ],
			[ 'wpe_status' => WPE_TRASHED ],
			'%d',
			'%d'
		);
	}

	/**
	 * process bulk actions of delete, restore, approve, cancel
	 *
	 * @param int $entry_status
	 * @param string $message
	 *
	 * @since 1.2.0
	 */
	public function wpe_process_bulk_action( $entry_status, $message ) {

		if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}
	
		$delete_arr = array();
		if ( isset( $_GET['bulk-delete'] ) ) {
			
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$bulk_delete_raw =  wp_unslash( $_GET['bulk-delete'] );
			if ( is_array( $bulk_delete_raw ) ) {
				$delete_arr = array_map( 'wpe_sanitize', $bulk_delete_raw );
			} else {

				$delete_arr[] = wpe_sanitize( $bulk_delete_raw );
			}
		}
	
		if ( ! empty( $delete_arr ) ) {
			foreach ( $delete_arr as $id ) {
				$this->update_registration_status( (int) $id, $entry_status );
			}
			$no_of_posts = count( $delete_arr );
			$message = $no_of_posts . $message;
			$this->wpe_admin_notice( $message );
		}
	}
	

	/**
     * Removes entries from display for deleted events.
     * 
     * @global object $wpdb instantiation of the wpdb class.
     *
     * @since 1.2.0
	 * @return bool
     */
	public function wpe_remove_deleted_event_entries(): bool {
        global $wpdb;
		$sql = "SELECT post_id FROM {$wpdb->prefix}$this->table_name";
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$results = $wpdb->get_results( $sql, ARRAY_A );
		$deleted_event = [];

		$args = array(
			'numberposts' => -1,
			'post_type'   => 'wp_events',
			'post_status' => array( 'publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash' ),
			'fields'	  => 'ids',
		);
		
		// phpcs:ignore WordPress.WP.DiscouragedFunctions.query_posts_query_posts
		$events = query_posts( $args );
		
		for( $i = 0; $i < sizeof( $results ); $i++ ) {
			foreach( $results[$i] as $postid => $id ) {
				if( !in_array( $id, $events ) ) {
					$deleted_event[] = (int) $id;
				}
			}
		}

		$deleted_event = array_unique( $deleted_event );
		$deleted_event = array_values( $deleted_event );

		if( !empty( $deleted_event ) ){
			for( $j = 0; $j < sizeof( $deleted_event ); $j++ ){
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$result = $wpdb->update(
					"{$wpdb->prefix}$this->table_name",
					['wpe_status' => -1],
					['post_id' => $deleted_event[$j] ],
					'%d',
					'%d'
				);
			}
			return $result;
		}
		return true;
	}

	
	
	

	/**
	 * Send notification to registrant if entry is approved/cancelled
	 *
	 * @param int $entry_status
	 * 
	 * @since 1.2.0
	 */
	public function wpe_approve_cancel_notification( $entry_status ) {
		$status          = $entry_status == WPE_CANCELLED ? 'cancelled.' : 'approved.';
		$append_message  = $entry_status == WPE_APPROVED ? 'We look forward to seeing you' : '';
	
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}
	
		$delete_arr = isset( $_GET['bulk-delete'] ) ? array_map( 'intval', (array) $_GET['bulk-delete'] ) : [];
		$mail_options = get_option('wpe_mail_settings');
		$firm_info 	  = get_option('wpe_firm_settings');
		$from_name    = $firm_info['mail_from_name'];
		$from_email   = $mail_options['mail_from'];
		$headers[]    = 'Content-Type: text/html;';
		$headers[]    = "from: $from_name <$from_email>";
	
		if ( is_array( $delete_arr ) ) {
			foreach ( $delete_arr as $id ) {
				$data	  = Wp_Events_Db_Actions::wpe_get_registration_data( $id, '', 'ARRAY_A' );
				Wpe_Shortcodes::set_form_data( $data[0] );
				$to       = $data[0]['email'];
				$subject  = 'Your registration for ' . get_the_title( $data[0]['post_id'] ) . ' is ' . $status;
				$message  = 'Dear ' . $data[0]['first_name'] . ' ' . $data[0]['last_name'] . ',<br />
					Thank you for registering for our upcoming Event. This is an auto-generated email to inform you that your registration for our upcoming Event is ' . $status . '<br />
					<br />
					The details of your registration are following.<br />
					[wpe_event_details] <br />
					[wpe_registration_details]<br />
					If you have any questions, please feel free to contact us at our office number or via email.<br />
					' . $append_message . '.<br />
					Sincerely,';
	
				$message = do_shortcode( $message, true );
				wp_mail( $to, html_entity_decode( $subject ), $message, $headers );
			}
		}
	}
	
}
